/*
 * Copyright (c) 2013-2023, Arm Limited and Contributors. All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef ARCH_HELPERS_H
#define ARCH_HELPERS_H

#include <inttypes.h>
#include "arch.h"

#define u_register_t uint64_t

/**********************************************************************
 * Macros which create inline functions to read or write CPU system
 * registers
 *********************************************************************/

#define _DEFINE_SYSREG_READ_FUNC(_name, _reg_name)				\
static inline u_register_t read_ ## _name(void)					\
{																\
	u_register_t v;												\
	__asm volatile ("mrs %0, " #_reg_name : "=r" (v));			\
	return v;													\
}

#define _DEFINE_SYSREG_WRITE_FUNC(_name, _reg_name)				\
static inline void write_ ## _name(u_register_t v)				\
{																\
	__asm volatile ("msr " #_reg_name ", %0" : : "r" (v));		\
	__asm volatile ("isb");										\
}

/* Define read function for system register */
#define DEFINE_SYSREG_READ_FUNC(_name) 			\
	_DEFINE_SYSREG_READ_FUNC(_name, _name)

/* Define read & write function for system register */
#define DEFINE_SYSREG_RW_FUNCS(_name)			\
	_DEFINE_SYSREG_READ_FUNC(_name, _name)		\
	_DEFINE_SYSREG_WRITE_FUNC(_name, _name)


/* AArch64 PMU registers */
DEFINE_SYSREG_RW_FUNCS(pmcr_el0)
DEFINE_SYSREG_RW_FUNCS(pmcntenset_el0)
DEFINE_SYSREG_RW_FUNCS(pmcntenclr_el0)
DEFINE_SYSREG_RW_FUNCS(pmselr_el0)
DEFINE_SYSREG_RW_FUNCS(pmccntr_el0)
DEFINE_SYSREG_RW_FUNCS(pmxevtyper_el0)
DEFINE_SYSREG_RW_FUNCS(pmccfiltr_el0)
DEFINE_SYSREG_RW_FUNCS(pmxevcntr_el0)
DEFINE_SYSREG_RW_FUNCS(pmovsclr_el0)

/* AArch64 secure registers */
DEFINE_SYSREG_RW_FUNCS(mdcr_el3)

/* AArch64 virtualization registers */
DEFINE_SYSREG_RW_FUNCS(mdcr_el2)

/* AArch64 secial-purpose registers */
DEFINE_SYSREG_READ_FUNC(CurrentEl)

static inline unsigned int get_current_el(void)
{
	return GET_EL(read_CurrentEl());
}

#endif /* ARCH_HELPERS_H */