/*
 * Copyright (c) 2013-2023, Arm Limited and Contributors. All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "arch_helpers.h"
#include "armv8_pmuv3_fn.h"
#include <stdio.h>

static void pmuv3_configCounter(struct pmu_event_selected* evt)
{
	int n, elx;
	uint64_t config = 0;

	elx = get_current_el();

	/* 
	 * Set the event filtering for current Exception level
	 * MODE_EL3: EL3
	 * MODE_EL2: None-secure EL2
	 */
	switch (elx) {
	case MODE_EL3:
		config = PMEVTYPERX_EL0_P_BIT | PMEVTYPERX_EL0_M_BIT;
		break;
	case MODE_EL2:
		config = PMEVTYPERX_EL0_NSH_BIT;
		break;
	}

	for (n = 0; evt[n].event.number != PMU_SELECTED_END; n++) {
		write_pmselr_el0(evt[n].cnt_idx);
		write_pmxevtyper_el0(config);
	}

	/* Set the event to be counted by each event counter */
	for (n = 0; evt[n].event.number != PMU_SELECTED_END; n++) {
		if (evt[n].event.number != CYCLE_COUNTER_EVENT) {
			write_pmselr_el0(evt[n].cnt_idx);
			write_pmxevtyper_el0((read_pmxevtyper_el0()			\
								& ~PMEVTYPERX_EL0_EVT_MASK) |	\
								evt[n].event.number);
		}
	}

	/* Set the initial counter value to zero */
	write_pmcr_el0(read_pmcr_el0() | PMCR_EL0_C_BIT);
	write_pmcr_el0(read_pmcr_el0() | PMCR_EL0_P_BIT);
}

static void pmuv3_enableCounter(struct pmu_event_selected* evt)
{
	int n;

	/* Enable each counter */
	for (n = 0; evt[n].event.number != PMU_SELECTED_END; n++) {
		write_pmcntenset_el0(read_pmcntenset_el0() | (1ULL << evt[n].cnt_idx));
	}

	/* Set the global counter enble bit */
	write_pmcr_el0(read_pmcr_el0() | PMCR_EL0_E_BIT);
}

static void pmuv3_disableCounter(struct pmu_event_selected* evt)
{
	int n;

	/* Set the global counter enble bit */
	write_pmcr_el0(read_pmcr_el0() & ~PMCR_EL0_E_BIT);

	/* Disable each counter */ 
	for (n = 0; evt[n].event.number != PMU_SELECTED_END; n++) {
		write_pmcntenclr_el0(read_pmcntenclr_el0() | (1ULL << evt[n].cnt_idx));
	}
}

static void pmuv3_snapshot(struct pmu_event_selected* evt)
{
	int n;

	/* Acquire the counting values for cycle counter and each event counters */
	for (n = 0; evt[n].event.number != PMU_SELECTED_END; n++) {

		evt[n].preval = evt[n].postval;

		if (evt[n].event.number != CYCLE_COUNTER_EVENT) {
			write_pmselr_el0(evt[n].cnt_idx);
			evt[n].postval = read_pmxevcntr_el0();
		} else {
			evt[n].postval = read_pmccntr_el0();
		}
	}
}

static void pmuv3_dumpResult(struct pmu_event_selected* evt)
{
	int n;

	printf("************************************************************\n");
	printf("               [armv8_pmuv3] Profiling Result               \n");
	printf("************************************************************\n");
	printf("PMU EVENT, PREVAL, POSTVAL, DELTA\n");
	for (n = 0; evt[n].event.number != PMU_SELECTED_END; n++) {
		printf("%s,%" PRIu64 ",%" PRIu64 ",%" PRIu64 "\n", evt[n].event.name, \
											evt[n].preval,                    \
											evt[n].postval,                   \
											evt[n].postval - evt[n].preval);
	}
	printf("************************************************************\n");
}

static int pmuv3_init(struct pmu_event_selected* evt)
{
	int nr_sel = 0;
	int cycle_flag = FALSE;
	int nr_sup = 0;
	int n, elx;

	/* 
	 * 1. Check the number of selected PMU events for event counters.
	 *	  If it exceeds the hardware supporting number, then stop profiling.
	 *	  If no PMU event is selected, then stop profiling.
	 * 2. Set the counter index for each selected PMU events.
	 *	  The counter index is used for using PMSELR_EL0 to select counter.
	 *	  0 - 30: To select event counter.
	 *	  31: To select cycle counter.
	 */
	for (n = 0; evt[n].event.number != PMU_SELECTED_END; n++) {
		if (evt[n].event.number != CYCLE_COUNTER_EVENT) {
			evt[n].cnt_idx = nr_sel;
			nr_sel++;
		} else {
			evt[n].cnt_idx = 31;
			cycle_flag = TRUE;
		}
	}

	nr_sup = (int)((read_pmcr_el0() & PMCR_EL0_N_BITS) >> PMCR_EL0_N_SHIFT);

	if (nr_sel > nr_sup) {
		printf("[armv8_pmuv3] ERROR: ");
		printf("Hardware supports %d PMU events for event counters ", nr_sup);
		printf("while %d events are selected!\r\n", nr_sel);
		return FALSE;
	} else if ((nr_sel == 0) && (cycle_flag==0)) {
		printf("[armv8_pmuv3] ERROR: No PMU event is selected! \r\n");
		return FALSE;
	}

	/* 
	 * Long cycle counter enable bit. 
	 * Even though Arm deprecates PMCR.LC = 0, it is still necessary to set
	 * PMCR.LC = 1 for Armv8.0 CPUs.
	 */ 
	write_pmcr_el0(read_pmcr_el0() | PMCR_EL0_LC_BIT); 

	/* 
	 * Additional system registers need to be set to enable event/cycle counter
	 * counting at EL2/EL3.
	 * 
	 * EL3: MDCR_EL3 control the PMU counters counting in secure state / at EL3.
	 *		1. Set .SPME = 1, to enable event counters counting in secure state.
	 * 		2. Set .SCCD = 1, to prohibit cycle counter counting in secure state.
	 * 		3. Set .MCCD = 1, to prohibit cycle counter counting at EL3.
	 * EL2: Enable the event counters in the second range (see MDCR_EL2.HPMN).
	 */
	elx = get_current_el();

	switch (elx) {
	case MODE_EL3:
		write_mdcr_el3(MDCR_SPME_BIT & ~(MDCR_MCCD_BIT | MDCR_SCCD_BIT));
		break;
	case MODE_EL2:
		write_mdcr_el2(MDCR_EL2_HPME_BIT);
		break;
	}
	return TRUE;
}

static void pmuv3_deinit(struct pmu_event_selected* evt)
{
	int n, elx;
	uint64_t overflow_status;

	/* 
	 * After the profiling, check the overflow status of each PMU counters.
	 * Pop up warning if the counter overflowed, then clear the overflow bit.
	 */
	overflow_status = read_pmovsclr_el0();

	for (n = 0; evt[n].event.number != PMU_SELECTED_END; n++) {
		if ((overflow_status & (1ULL << evt[n].cnt_idx)) != 0) {
			printf("[armv8_pmuv3] WARNING: ");
			printf("Counter for %s overflowed in this profiling!\r\n",\
					evt[n].event.name);
			write_pmovsclr_el0(1ULL << evt[n].cnt_idx);
		} 
	}

	/*
	 * After the profiling, recover the system registers setting.
	 * EL3: To prohibit event counters and cycle counter counting at EL3.
	 * EL2: To prohibit event counters in the second range counting at EL2.
	 */
	elx = get_current_el();

	switch (elx) {
	case MODE_EL3:
		write_mdcr_el3((read_mdcr_el3() & ~MDCR_SPME_BIT) |\
						(MDCR_MCCD_BIT | MDCR_SCCD_BIT));
		break;
    case MODE_EL2:
		write_mdcr_el2(read_mdcr_el2() & ~MDCR_EL2_HPME_BIT);
		break;
	}
}

static int isProfiling = FALSE;

void pmuv3_startProfiling(struct pmu_event_selected* evt)
{
	if(pmuv3_init(evt) == FALSE) {
		isProfiling = FALSE;
		return;
	}

	pmuv3_configCounter(evt);
	pmuv3_enableCounter(evt);
	pmuv3_snapshot(evt);
	isProfiling = TRUE;
}

void pmuv3_stopProfiling(struct pmu_event_selected* evt)
{
	if (isProfiling == TRUE) {
		pmuv3_disableCounter(evt);
		pmuv3_snapshot(evt);
		pmuv3_dumpResult(evt);
		pmuv3_deinit(evt);
		isProfiling = FALSE;
	}
}