#!/usr/bin/env python3

""" Convert JSON PMU events to C code for specified CPU. """
import json
import os
import subprocess
import argparse

REPO_URL = 'https://github.com/ARM-software/data.git'
LOCAL_PATH = os.path.dirname(os.path.abspath(__file__))

def repo_init(repo_url: str, download_path: str) -> list:

	""" Initialize the repository and return a list of supported CPU names. """
	repo_name = repo_url.split("/")[-1].split(".")[0]
	repo_local_path = os.path.join(download_path, repo_name)

	if not os.path.exists(repo_local_path):    
		subprocess.call(["git", "clone", repo_url, repo_local_path])
    
	all_files = os.listdir(os.path.join(repo_local_path, 'pmu'))
	file_names = [os.path.splitext(file)[0] for file in all_files]
	return file_names

def parse_json(json_file: str, c_file: str) -> None:

	""" Parse the JSON file and generate C code. """
	try:
		with open(json_file, 'r') as input_file:
			data = json.load(input_file)
			events = data.get("events", [])
			name_code_array = [{"name": event.get("name", hex(event["code"])), "number": event["code"]} for event in events]

		with open(c_file, 'w') as output_file:
			output_file.write("#include \"armv8_pmuv3_events.h\"\r\n")
			output_file.write("/* Selected PMU Events Table */\n")
			output_file.write("struct pmu_event_selected evt_select[] = {\n")
			output_file.write("    { .event.name = \"INST_RETIRED\", .event.number = 0x08 },\n")
			output_file.write("    { .event.name = \"CYCLES\", .event.number = CYCLE_COUNTER_EVENT }, \n")
			output_file.write("    /* Always place the following one at the end of the table */\n")
			output_file.write("    { .event.name = \"NULL\", .event.number = PMU_SELECTED_END } \n")
			output_file.write("};\r\n")
			output_file.write("/* PMU Events Mapping Table */\n")
			output_file.write(f"const struct pmu_event pmu_events_map[] = {{\n")
			output_file.write("    /* Supporting PMU events for event counter */ \n")
			output_file.writelines([f'    {{ .name = "{nc["name"]}", .number = {hex(nc["number"])} }},\n' for nc in name_code_array])
			output_file.write("    /* Supporting event for cycle counter */ \n")
			output_file.write("    { .name = \"CYCLES\", .number = CYCLE_COUNTER_EVENT } \n")
			output_file.write("};\n")

		print(c_file + " is created.")
   
	except FileNotFoundError:
		print(f"JSON file '{json_file}' does not exist!")

if __name__ == '__main__':

	cpulist = repo_init(REPO_URL, LOCAL_PATH)
    
	parser = argparse.ArgumentParser(description='Convert PMU events in JSON format to C code for specified CPU.')
	parser.add_argument('--cpu', type=str, help='CPU name')
	parser.add_argument('--list', action='store_true', help='Display the list of supporting CPU names')

	args = parser.parse_args()

	if args.list:
		print(','.join(cpulist))
	elif args.cpu:
		json_file = os.path.join(LOCAL_PATH, "data", "pmu", args.cpu + ".json")
		c_file = os.path.join(LOCAL_PATH, "armv8_pmuv3_events.c")
		parse_json(json_file,c_file)
	else:
		parser.print_help()
